/* 

	Boot Writer Module for updating the bootloader itself! 
	By Mauro Grassi, 2010.
	
*/

#include "flash.h"
#include "led.h"
#include "boot.h"
#include "Compiler.h"
#include "HardwareProfile.h"
#include "GenericTypeDefs.h"
#include <math.h>
#include "usb_device.h"
#include "usb.h"
#include "usb_function_generic.h"
#include "usb_config.h"
#include "bootwriter.h"
#include "main.h"

#if(BOOT_WRITER)

void _boothook(void);

#pragma udata 

unsigned int  bwoff;
unsigned char bwtimeout;
unsigned char bwprescale;

#pragma code serialbootloadercode=0x1D400

/* Make sure that the entry point for the serial bootloader is at a fixed address */

unsigned char doBootLoaderUpdate(void)
{
	/* 
		This is simple user code to update the bootloader through the serial port! 
		Avoids using the USB and is self contained, simple and small code...
		Mauro Grassi, December 2010...	
	
	*/
	unsigned char   c;
	unsigned long   startAddress;
	unsigned long	endAddress;
	unsigned long	numBytes;
	unsigned long   done;
 	unsigned int	data;
 	unsigned int 	pll_startup_counter;
	
	INTCON=0;
	pll_startup_counter=600;
    OSCTUNEbits.PLLEN = 1;  //Enable the PLL and wait 2+ms until the PLL locks before enabling USB module
    while(pll_startup_counter--);
   	
    ANCON1=0x0F;
	ANCON0=0xFF;
    TRISA=0xFF;
    TRISB=0xFF;
    TRISC=0xFF;
    
    RPOR0=0;
    RPOR1=0;
    RPOR2=0;
    RPOR3=0;
    RPOR4=0;
    RPOR5=0;
    RPOR6=0;
    RPOR7=0;
    RPOR8=0;
    RPOR9=0;
    RPOR10=0;
    RPOR11=0;
    RPOR12=0;
    RPOR13=0;
    RPOR17=0;
    RPOR18=0;
 
	#if(USE_PMDIS)
		PMDIS3=0xFF;
		PMDIS2=0xFF;
		PMDIS1=0xFF;
		PMDIS0=0xFF;
	#endif  
				
	LED1=!LED1_ON;
	LED1_TRIS=0;
	done=0;
	/* Initialize the serial port */
	PMDIS0bits.UART2MD=0;	
	TRISAbits.TRISA0=1;
	ANCON0bits.PCFG0=1;
	TRISAbits.TRISA1=1;
	ANCON0bits.PCFG1=1;
	RCSTA2=0;
	TXSTA2=0;
	BAUDCON2=0;
	ODCON2bits.U2OD=0;
	BAUDCON2bits.RXDTP=1;
	BAUDCON2bits.TXCKP=1;
	BAUDCON2bits.BRG16=1;
	TXSTA2bits.BRGH=1;
	/* Baud Rate */
	SPBRGH2=(unsigned char)0;
	SPBRG2=(unsigned char)103;		
	TXSTA2bits.TXEN=1;
	PIR3bits.RC2IF=0;
	RCSTA2bits.CREN=1;		
	RCSTA2bits.SPEN=1;
	RPOR1=6;
	RPINR16=0;
	TRISAbits.TRISA1=0;
	/* Serial Port Is Now Enabled! */
	bwprescale=0;
	bwoff=0;
	bwtimeout=BW_TIMEOUT_LONG;
	while(1)
	{
		/* Enter The Main Loop */

		/* Wait For a Character */
		(void)getSerialCharsBootloader(&c, 1, MAGIC_BYTE_COMMAND_ECHO, 1);
		switch(c)
		{
			default:
					break;
	     	
	     	case CMD_BW_RESYNC:
	     			bwprescale=0;
	     			bwoff=0;
	     			bwtimeout=BW_TIMEOUT_LONG;
	     			break;
	     	
	     	case CMD_BW_GET_DONE:
	     			putSerialCharsBootloader((unsigned char*)&done, 4);
	        		break;
	        		
	     	case CMD_BW_SET_FLASH_ADDRESS:
	    			(void)getSerialCharsBootloader((unsigned char*)&startAddress, 4, 0, 1);
	     			break;
	        
	        case CMD_BW_SET_END_ADDRESS:
	        		(void)getSerialCharsBootloader((unsigned char*)&endAddress, 4, 0, 1);
	        		break;
	        		   				
	        case CMD_BW_READ_MEMORY:
	          		bwprescale=0;
	     			bwoff=0;
	     			bwtimeout=BW_TIMEOUT_LONG;
	     			done=ReadFlashOverSerial((unsigned long)startAddress, (unsigned long)endAddress);
	        		break;
	         
	         case CMD_BW_WRITE_MEMORY:
	         		bwprescale=0;
	     			bwoff=0;
	     			bwtimeout=BW_TIMEOUT_LONG;
	     			(void)getSerialCharsBootloader((unsigned char*)&tempString[0], 64, 0, 0);
					(void)WriteFlashSerial(startAddress, 64, &tempString[0]);
					c=0x00;
					(void)putSerialCharsBootloader((unsigned char*)&c, 1);
					startAddress+=64;
					break;
						
			case CMD_BW_WRITE_WORD:
			 		(void)getSerialCharsBootloader((unsigned char*)&tempString[0], 2, 0, 0);
					data=(unsigned int)tempString[0];
					data+=(unsigned int)((unsigned int)tempString[1]<<8);
					(void)WriteWordFlashSerial(startAddress, data);
					c=0x00;
					(void)putSerialCharsBootloader((unsigned char*)&c, 1);
					startAddress+=2;
					break;
     		 
			case CMD_BW_ERASE_MEMORY:
					done=EraseFlashOverSerial((unsigned long)startAddress, (unsigned long)endAddress);
					c=0x00;
					(void)putSerialCharsBootloader((unsigned char*)&c, 1);
					break;
			
			case CMD_BW_BOOT_RESET:
					bootFunction=1;
				    _asm goto _boothook _endasm
					break;
			}
			
	 } 
	 return 0;
}

void idleTaskSerialBootloader(void)
{
	if(bwprescale>0)
	{
		bwprescale--;
	}
	else
	{
		bwprescale=bwtimeout;
		if(bwoff>0)
		{
			bwoff--;
		}
		else
		{
			LED1=!LED1;
			if(LED1==LED1_ON)
			{
				bwoff=BW_LED_ON_TIMEOUT;
			}
			else
			{
				bwoff=BW_LED_OFF_TIMEOUT;
			}
		}
	}
}

unsigned char getSerialCharsBootloader(unsigned char* in, unsigned int num, unsigned char reply, unsigned char echo)
{
	unsigned char c;

	while(num>0)
	{
		idleTaskSerialBootloader();
		if(PIR3bits.RC2IF)
		{
			PIR3bits.RC2IF=0;
			if(RCSTA2bits.OERR)
			{
				RCSTA2bits.CREN=0;
				RCSTA2bits.CREN=1;	
				c=RCREG2;
				if(echo)TXREG2=0xFF;
			}
			else
			{
				c=RCREG2;
				while((RCSTA2bits.SPEN)&&(TXSTA2bits.TXEN)&&(TXSTA2bits.TRMT==0));
				if(echo)
				{
					if(reply)TXREG2=reply; else TXREG2=c;
				}	
				*in++=c;
			}		
			num--;
			/* Echo Reply */
		}
	}	
	return 1;
}

unsigned char putSerialCharsBootloader(unsigned char* in, unsigned int num)
{
	while(num>0)
	{
		idleTaskSerialBootloader();
		while((RCSTA2bits.SPEN)&&(TXSTA2bits.TXEN)&&(TXSTA2bits.TRMT==0));
		TXREG2=*in++;
		num--;
	}
	return 1;
}

unsigned long EraseFlashOverSerial(unsigned long startaddr, unsigned long endaddr)
{
		DWORD_VAL flash_addr;
		unsigned long done;

		done=0;
		while(startaddr<endaddr)
		{
			idleTaskSerialBootloader();
			
			flash_addr.Val = startaddr;
			TBLPTRU = flash_addr.byte.UB;						//Load the address to Address pointer registers
			TBLPTRH = flash_addr.byte.HB;	
			TBLPTRL	= flash_addr.byte.LB;
			//*********Flash Erase sequence*****************
			EECON1bits.WREN = 1;
			EECON1bits.FREE = 1;
			EECON2 = 0x55;
			EECON2 = 0xAA;
			EECON1bits.WR = 1;
			startaddr = startaddr + FLASH_ERASE_BLOCK;
			done+=(unsigned long)FLASH_ERASE_BLOCK;
		}
		EECON1bits.WREN=0;
		return done;
}

unsigned long ReadFlashOverSerial(unsigned long startaddr, unsigned long endaddr)
{
		DWORD_VAL flash_addr;
		unsigned long   done;
		
		flash_addr.Val = startaddr;
		TBLPTRU = flash_addr.byte.UB;						//Load the address to Address pointer registers
		TBLPTRH = flash_addr.byte.HB;	
		TBLPTRL	= flash_addr.byte.LB;
		done=0;
		while(startaddr<endaddr)
		{
			//*********** Table read sequence ******************************
			idleTaskSerialBootloader();
			while((RCSTA2bits.SPEN)&&(TXSTA2bits.TXEN)&&(TXSTA2bits.TRMT==0));
			_asm	TBLRDPOSTINC _endasm
			TXREG2=TABLAT;
			startaddr++;
			done++;
		}	
		return done;
}

void WriteFlashSerial(unsigned long startaddr, unsigned int num_bytes, unsigned char *flash_array)
{
		unsigned char write_byte=0;
		DWORD_VAL flash_addr;
		unsigned long done;
		
		flash_addr.Val = startaddr;

		startaddr &= FLASH_WRITE_BLOCK_ALIGN_MASK;
		startaddr += FLASH_WRITE_BLOCK ;
		write_byte = startaddr - flash_addr.Val;
		done=0;
		while(num_bytes)
		{
				TBLPTRU = flash_addr.byte.UB;						//Load the address to Address pointer registers
				TBLPTRH = flash_addr.byte.HB;	
				TBLPTRL	= flash_addr.byte.LB;
				while(write_byte--)
				{
					TABLAT = *flash_array++;
					_asm  TBLWTPOSTINC 	_endasm
					idleTaskSerialBootloader();
					if(--num_bytes==0)	break;
				}

				TBLPTRU = flash_addr.byte.UB;						//Load the address to Address pointer registers
				TBLPTRH = flash_addr.byte.HB;	
				TBLPTRL	= flash_addr.byte.LB;
			  //*********** Flash write sequence ***********************************
			  EECON1bits.WREN = 1;
			  EECON1bits.WPROG=0;
			  EECON2 = 0x55;
			  EECON2 = 0xAA;
			  EECON1bits.WR =1;
			  EECON1bits.WREN = 0 ; 
			 write_byte = FLASH_WRITE_BLOCK;
			 flash_addr.Val = flash_addr.Val + FLASH_WRITE_BLOCK;									//increment to one block of 64 bytes
		}
		EECON1bits.WREN=0;
}

void WriteWordFlashSerial(unsigned long startaddr, unsigned int data)
{
	/* 
	
	Write a single word (2 bytes) to flash (does not need previous erase operation...)
	
	*/

			DWORD_VAL flash_addr;
			WORD_VAL flash_data;

			flash_addr.Val = startaddr;
			flash_data.Val = data;
			
			TBLPTRU = flash_addr.byte.UB;						//Load the address to Address pointer registers
			TBLPTRH = flash_addr.byte.HB;	
			TBLPTRL	= flash_addr.byte.LB;
				
			TABLAT = flash_data.byte.LB;
			_asm  TBLWTPOSTINC 	_endasm

			TABLAT =flash_data.byte.HB;
			_asm  TBLWT 	_endasm
	
			TBLPTRU = flash_addr.byte.UB;						//Load the address to Address pointer registers
			TBLPTRH = flash_addr.byte.HB;	
			TBLPTRL	= flash_addr.byte.LB;
		  
		  //*********** Flash write sequence ***********************************
			EECON1bits.WPROG = 1;	
			EECON1bits.WREN = 1;
			//EECON1bits.FREE = 1;
			EECON2  = 0x55;
			EECON2  = 0xAA;
			EECON1bits.WR = 1;
			EECON1bits.WREN = 0 ;
			EECON1bits.WPROG=0;
}

#endif
